import tkinter as tk
from tkinter import ttk
import math
import random
import time
from threading import Thread

class GasSimulation:
    def __init__(self, root):
        self.root = root
        self.root.title("Визуализация поведения идеального газа")
        self.root.geometry("1200x800")
        
        # Параметры системы
        self.temperature = 300  # K
        self.pressure = 101325  # Па
        self.volume = 0.001     # m³
        self.molecule_count = 50
        self.gas_type = "Одноатомный"
        
        # Физические константы
        self.k = 1.380649e-23  # Постоянная Больцмана
        self.R = 8.314         # Универсальная газовая постоянная
        
        # Состояние симуляции
        self.simulating = False
        self.molecules = []
        self.container_size = 400
        self.time_step = 0.01
        
        # Статистика
        self.collisions = 0
        self.kinetic_energy = 0
        self.pressure_calculated = 0
        
        self.setup_ui()
        self.initialize_molecules()
    
    def setup_ui(self):
        # Основной фрейм
        main_frame = ttk.Frame(self.root, padding="10")
        main_frame.pack(fill=tk.BOTH, expand=True)
        
        # Левая панель - управление
        left_frame = ttk.Frame(main_frame, width=300)
        left_frame.pack(side=tk.LEFT, fill=tk.Y, padx=(0, 10))
        left_frame.pack_propagate(False)
        
        # Правая панель - визуализация и статистика
        right_frame = ttk.Frame(main_frame)
        right_frame.pack(side=tk.RIGHT, fill=tk.BOTH, expand=True)
        
        # === Левая панель - параметры газа ===
        ttk.Label(left_frame, text="Параметры газа", font=("Arial", 12, "bold")).pack(anchor=tk.W, pady=(0, 10))
        
        # Тип газа
        gas_frame = ttk.LabelFrame(left_frame, text="Тип газа", padding="5")
        gas_frame.pack(fill=tk.X, pady=5)
        
        self.gas_var = tk.StringVar(value=self.gas_type)
        gases = ["Одноатомный", "Двухатомный", "Многоатомный"]
        
        for gas in gases:
            ttk.Radiobutton(gas_frame, text=gas, variable=self.gas_var, 
                           value=gas, command=self.on_gas_type_change).pack(anchor=tk.W)
        
        # Параметры системы
        params_frame = ttk.LabelFrame(left_frame, text="Параметры системы", padding="5")
        params_frame.pack(fill=tk.X, pady=5)
        
        # Температура
        ttk.Label(params_frame, text="Температура (K):").pack(anchor=tk.W)
        self.temp_var = tk.DoubleVar(value=self.temperature)
        temp_scale = ttk.Scale(params_frame, from_=100, to=1000, variable=self.temp_var,
                              orient=tk.HORIZONTAL, command=self.on_temp_change)
        temp_scale.pack(fill=tk.X, pady=2)
        self.temp_label = ttk.Label(params_frame, text=f"{self.temperature} K")
        self.temp_label.pack(anchor=tk.W)
        
        # Давление
        ttk.Label(params_frame, text="Давление (кПа):").pack(anchor=tk.W, pady=(10, 0))
        self.pressure_var = tk.DoubleVar(value=self.pressure / 1000)
        pressure_scale = ttk.Scale(params_frame, from_=10, to=500, variable=self.pressure_var,
                                  orient=tk.HORIZONTAL, command=self.on_pressure_change)
        pressure_scale.pack(fill=tk.X, pady=2)
        self.pressure_label = ttk.Label(params_frame, text=f"{self.pressure/1000:.1f} кПа")
        self.pressure_label.pack(anchor=tk.W)
        
        # Объем
        ttk.Label(params_frame, text="Объем сосуда:").pack(anchor=tk.W, pady=(10, 0))
        self.volume_var = tk.DoubleVar(value=1.0)
        volume_scale = ttk.Scale(params_frame, from_=0.5, to=3.0, variable=self.volume_var,
                                orient=tk.HORIZONTAL, command=self.on_volume_change)
        volume_scale.pack(fill=tk.X, pady=2)
        self.volume_label = ttk.Label(params_frame, text="Нормальный")
        self.volume_label.pack(anchor=tk.W)
        
        # Количество молекул
        ttk.Label(params_frame, text="Количество молекул:").pack(anchor=tk.W, pady=(10, 0))
        self.mol_count_var = tk.IntVar(value=self.molecule_count)
        mol_scale = ttk.Scale(params_frame, from_=10, to=200, variable=self.mol_count_var,
                             orient=tk.HORIZONTAL, command=self.on_mol_count_change)
        mol_scale.pack(fill=tk.X, pady=2)
        self.mol_count_label = ttk.Label(params_frame, text=f"{self.molecule_count} молекул")
        self.mol_count_label.pack(anchor=tk.W)
        
        # Управление симуляцией
        control_frame = ttk.LabelFrame(left_frame, text="Управление", padding="5")
        control_frame.pack(fill=tk.X, pady=5)
        
        self.start_button = ttk.Button(control_frame, text="Запуск", 
                                      command=self.start_simulation)
        self.start_button.pack(fill=tk.X, pady=2)
        
        self.stop_button = ttk.Button(control_frame, text="Стоп", 
                                     command=self.stop_simulation, state=tk.DISABLED)
        self.stop_button.pack(fill=tk.X, pady=2)
        
        self.reset_button = ttk.Button(control_frame, text="Сброс", 
                                      command=self.reset_simulation)
        self.reset_button.pack(fill=tk.X, pady=2)
        
        # Информация о газе
        info_frame = ttk.LabelFrame(left_frame, text="Свойства газа", padding="5")
        info_frame.pack(fill=tk.X, pady=5)
        
        self.gas_info_text = tk.Text(info_frame, height=6, width=30, font=("Arial", 9))
        self.gas_info_text.pack(fill=tk.BOTH)
        self.update_gas_info()
        
        # === Правая панель - визуализация ===
        # Холст для отрисовки
        canvas_frame = ttk.Frame(right_frame)
        canvas_frame.pack(fill=tk.BOTH, expand=True, pady=(0, 10))
        
        self.canvas = tk.Canvas(canvas_frame, width=600, height=500, 
                               bg="black", highlightthickness=1, highlightbackground="gray")
        self.canvas.pack(fill=tk.BOTH, expand=True)
        
        # Статистика в реальном времени
        stats_frame = ttk.LabelFrame(right_frame, text="Статистика в реальном времени", padding="5")
        stats_frame.pack(fill=tk.X)
        
        stats_grid = ttk.Frame(stats_frame)
        stats_grid.pack(fill=tk.X)
        
        # Статистика слева
        stats_left = ttk.Frame(stats_grid)
        stats_left.pack(side=tk.LEFT, fill=tk.X, expand=True)
        
        self.energy_label = ttk.Label(stats_left, text="Средняя кинетическая энергия: 0 J", 
                                     font=("Arial", 10))
        self.energy_label.pack(anchor=tk.W)
        
        self.speed_label = ttk.Label(stats_left, text="Средняя скорость: 0 m/s", 
                                    font=("Arial", 10))
        self.speed_label.pack(anchor=tk.W)
        
        self.collisions_label = ttk.Label(stats_left, text="Столкновений: 0", 
                                         font=("Arial", 10))
        self.collisions_label.pack(anchor=tk.W)
        
        # Статистика справа
        stats_right = ttk.Frame(stats_grid)
        stats_right.pack(side=tk.RIGHT, fill=tk.X, expand=True)
        
        self.pressure_label_live = ttk.Label(stats_right, text="Рассчитанное давление: 0 кПа", 
                                           font=("Arial", 10))
        self.pressure_label_live.pack(anchor=tk.W)
        
        self.temp_label_live = ttk.Label(stats_right, text="Рассчитанная температура: 0 K", 
                                        font=("Arial", 10))
        self.temp_label_live.pack(anchor=tk.W)
        
        self.volume_label_live = ttk.Label(stats_right, text="Молярный объем: 0 m³/mol", 
                                          font=("Arial", 10))
        self.volume_label_live.pack(anchor=tk.W)
    
    def on_gas_type_change(self):
        self.gas_type = self.gas_var.get()
        self.update_gas_info()
        if self.simulating:
            self.initialize_molecules()
    
    def on_temp_change(self, event=None):
        self.temperature = self.temp_var.get()
        self.temp_label.config(text=f"{self.temperature:.0f} K")
        if self.simulating:
            self.adjust_temperature()
    
    def on_pressure_change(self, event=None):
        self.pressure = self.pressure_var.get() * 1000
        self.pressure_label.config(text=f"{self.pressure/1000:.1f} кПа")
    
    def on_volume_change(self, event=None):
        volume_scale = self.volume_var.get()
        if volume_scale < 1.0:
            self.volume_label.config(text="Малый")
        elif volume_scale > 2.0:
            self.volume_label.config(text="Большой")
        else:
            self.volume_label.config(text="Нормальный")
        
        self.volume = 0.001 * volume_scale
        if self.simulating:
            self.resize_container()
    
    def on_mol_count_change(self, event=None):
        self.molecule_count = self.mol_count_var.get()
        self.mol_count_label.config(text=f"{self.molecule_count} молекул")
        if self.simulating:
            self.initialize_molecules()
    
    def update_gas_info(self):
        self.gas_info_text.delete(1.0, tk.END)
        
        if self.gas_type == "Одноатомный":
            info = """Одноатомный газ (Ar, He, Ne)
Степени свободы: 3
Cv = 3/2 R
Cp = 5/2 R
γ = 5/3 ≈ 1.67"""
        elif self.gas_type == "Двухатомный":
            info = """Двухатомный газ (N₂, O₂, H₂)
Степени свободы: 5
Cv = 5/2 R
Cp = 7/2 R
γ = 7/5 = 1.4"""
        else:  # Многоатомный
            info = """Многоатомный газ (CO₂, H₂O)
Степени свободы: 6
Cv = 3 R
Cp = 4 R
γ = 4/3 ≈ 1.33"""
        
        self.gas_info_text.insert(1.0, info)
    
    def get_degrees_of_freedom(self):
        if self.gas_type == "Одноатомный":
            return 3
        elif self.gas_type == "Двухатомный":
            return 5
        else:  # Многоатомный
            return 6
    
    def initialize_molecules(self):
        self.molecules = []
        container_margin = 50
        container_size = self.container_size - 2 * container_margin
        
        for i in range(self.molecule_count):
            # Случайная позиция внутри контейнера
            x = random.uniform(container_margin, container_margin + container_size)
            y = random.uniform(container_margin, container_margin + container_size)
            
            # Скорость based on temperature
            avg_speed = math.sqrt(3 * self.k * self.temperature / (1e-26))  # Упрощенная формула
            speed = random.uniform(0.5 * avg_speed, 1.5 * avg_speed)
            angle = random.uniform(0, 2 * math.pi)
            vx = speed * math.cos(angle)
            vy = speed * math.sin(angle)
            
            # Цвет в зависимости от типа газа
            if self.gas_type == "Одноатомный":
                color = "lightblue"
                radius = 4
            elif self.gas_type == "Двухатомный":
                color = "lightgreen" 
                radius = 5
            else:  # Многоатомный
                color = "lightcoral"
                radius = 6
            
            molecule = {
                'x': x, 'y': y,
                'vx': vx, 'vy': vy,
                'radius': radius,
                'color': color,
                'mass': 1e-26  # Упрощенная масса
            }
            self.molecules.append(molecule)
    
    def resize_container(self):
        # Изменение размера контейнера при изменении объема
        scale_factor = math.sqrt(self.volume_var.get())
        new_size = int(400 * scale_factor)
        self.container_size = min(new_size, 500)
    
    def adjust_temperature(self):
        # Корректировка скоростей молекул при изменении температуры
        temp_factor = math.sqrt(self.temperature / 300)  # 300K как база
        
        for molecule in self.molecules:
            molecule['vx'] *= temp_factor
            molecule['vy'] *= temp_factor
    
    def start_simulation(self):
        if not self.simulating:
            self.simulating = True
            self.start_button.config(state=tk.DISABLED)
            self.stop_button.config(state=tk.NORMAL)
            
            thread = Thread(target=self.run_simulation)
            thread.daemon = True
            thread.start()
    
    def stop_simulation(self):
        self.simulating = False
        self.start_button.config(state=tk.NORMAL)
        self.stop_button.config(state=tk.DISABLED)
    
    def reset_simulation(self):
        self.simulating = False
        self.collisions = 0
        self.kinetic_energy = 0
        self.initialize_molecules()
        self.draw_scene()
        self.start_button.config(state=tk.NORMAL)
        self.stop_button.config(state=tk.DISABLED)
    
    def run_simulation(self):
        while self.simulating:
            self.update_physics()
            self.draw_scene()
            self.update_statistics()
            time.sleep(0.02)
    
    def update_physics(self):
        container_margin = 50
        container_size = self.container_size - 2 * container_margin
        
        total_ke = 0
        self.collisions = 0
        
        for molecule in self.molecules:
            # Обновление позиции
            molecule['x'] += molecule['vx'] * self.time_step
            molecule['y'] += molecule['vy'] * self.time_step
            
            # Столкновение со стенками
            if molecule['x'] - molecule['radius'] < container_margin:
                molecule['x'] = container_margin + molecule['radius']
                molecule['vx'] = abs(molecule['vx'])
                self.collisions += 1
            elif molecule['x'] + molecule['radius'] > container_margin + container_size:
                molecule['x'] = container_margin + container_size - molecule['radius']
                molecule['vx'] = -abs(molecule['vx'])
                self.collisions += 1
            
            if molecule['y'] - molecule['radius'] < container_margin:
                molecule['y'] = container_margin + molecule['radius']
                molecule['vy'] = abs(molecule['vy'])
                self.collisions += 1
            elif molecule['y'] + molecule['radius'] > container_margin + container_size:
                molecule['y'] = container_margin + container_size - molecule['radius']
                molecule['vy'] = -abs(molecule['vy'])
                self.collisions += 1
            
            # Кинетическая энергия
            speed_sq = molecule['vx']**2 + molecule['vy']**2
            total_ke += 0.5 * molecule['mass'] * speed_sq
        
        self.kinetic_energy = total_ke
        
        # Простое обнаружение столкновений между молекулами
        for i in range(len(self.molecules)):
            for j in range(i + 1, len(self.molecules)):
                mol1 = self.molecules[i]
                mol2 = self.molecules[j]
                
                dx = mol1['x'] - mol2['x']
                dy = mol1['y'] - mol2['y']
                distance = math.sqrt(dx**2 + dy**2)
                
                if distance < mol1['radius'] + mol2['radius']:
                    # Простое упругое столкновение
                    if distance > 0:
                        # Обмен скоростями (упрощенная модель)
                        mol1['vx'], mol2['vx'] = mol2['vx'], mol1['vx']
                        mol1['vy'], mol2['vy'] = mol2['vy'], mol1['vy']
                        self.collisions += 1
    
    def draw_scene(self):
        self.canvas.delete("all")
        
        # Рисование контейнера
        container_margin = 50
        container_size = self.container_size
        
        self.canvas.create_rectangle(
            container_margin, container_margin,
            container_margin + container_size, container_margin + container_size,
            outline="white", width=2
        )
        
        # Рисование молекул
        for molecule in self.molecules:
            x, y = molecule['x'], molecule['y']
            radius = molecule['radius']
            color = molecule['color']
            
            self.canvas.create_oval(
                x - radius, y - radius,
                x + radius, y + radius,
                fill=color, outline=color
            )
            
            # Для двухатомных и многоатомных - дополнительная визуализация
            if self.gas_type == "Двухатомный":
                # Рисуем связь между атомами
                angle = math.atan2(molecule['vy'], molecule['vx'])
                bond_length = radius * 1.5
                x2 = x + bond_length * math.cos(angle)
                y2 = y + bond_length * math.sin(angle)
                self.canvas.create_line(x, y, x2, y2, fill=color, width=2)
                self.canvas.create_oval(x2-2, y2-2, x2+2, y2+2, fill=color, outline=color)
            
            elif self.gas_type == "Многоатомный":
                # Рисуем треугольник для многоатомных молекул
                size = radius * 1.2
                points = [
                    x, y - size,
                    x - size * 0.866, y + size * 0.5,
                    x + size * 0.866, y + size * 0.5
                ]
                self.canvas.create_polygon(points, fill=color, outline=color)
        
        # Информация о контейнере
        info_text = f"Температура: {self.temperature:.0f} K\n"
        info_text += f"Давление: {self.pressure/1000:.1f} кПа\n"
        info_text += f"Объем: {self.volume*1000:.1f} л\n"
        info_text += f"Молекул: {self.molecule_count}"
        
        self.canvas.create_text(
            container_margin + container_size + 10, container_margin + 20,
            text=info_text, fill="white", anchor=tk.W, font=("Arial", 10)
        )
    
    def update_statistics(self):
        # Расчет статистики
        total_mass = sum(mol['mass'] for mol in self.molecules)
        total_speed_sq = sum(mol['vx']**2 + mol['vy']**2 for mol in self.molecules)
        avg_speed = math.sqrt(total_speed_sq / len(self.molecules)) if self.molecules else 0
        
        # Расчет давления (упрощенная формула)
        area = 4 * (self.container_size - 100)  # Периметр контейнера
        self.pressure_calculated = (self.collisions * total_mass * avg_speed) / (area * self.time_step) if area > 0 else 0
        
        # Расчет температуры из кинетической теории
        dof = self.get_degrees_of_freedom()
        temp_calculated = (2 * self.kinetic_energy) / (dof * len(self.molecules) * self.k) if self.molecules else 0
        
        # Молярный объем
        molar_volume = self.volume / (len(self.molecules) * 1.66e-24)  # Упрощенный расчет
        
        # Обновление меток
        self.energy_label.config(text=f"Средняя кинетическая энергия: {self.kinetic_energy/len(self.molecules):.2e} J")
        self.speed_label.config(text=f"Средняя скорость: {avg_speed:.1f} m/s")
        self.collisions_label.config(text=f"Столкновений/кадр: {self.collisions}")
        
        self.pressure_label_live.config(text=f"Рассчитанное давление: {self.pressure_calculated/1000:.1f} кПа")
        self.temp_label_live.config(text=f"Рассчитанная температура: {temp_calculated:.1f} K")
        self.volume_label_live.config(text=f"Молярный объем: {molar_volume:.2e} m³/mol")

def main():
    root = tk.Tk()
    app = GasSimulation(root)
    root.mainloop()

if __name__ == "__main__":
    main()
